/* -- AIX/6000 System monitor
**
**     get_nfsstat.c
**
** Copyright (c) 1991-1995 Jussi Maki, All Rights Reserved.
** Copyright (c) 1993-2001 Marcel Mol, All Rights Reserved.
** NON-COMMERCIAL USE ALLOWED. YOU ARE FREE TO DISTRIBUTE
** THIS PROGRAM AND MODIFY IT AS LONG AS YOU KEEP ORIGINAL
** COPYRIGHTS.
*/

#include <stdio.h>      /* for perror */
#include <nlist.h>
#include "config.h"
#include "get_nfsstat.h"
#include "getkmemdata.h"

#if USE_NFS_CNTL == 1
# define NFS_EXT_PATH "/usr/lib/drivers/nfs.ext"
# include <sys/types.h>
# ifdef DO_SYSCONFIG
#  include <sys/sysconfig.h>
# endif
# include <dlfcn.h>

typedef   void (*SimpleFunc) ();

SimpleFunc my_nfs_cntl;

#else

# define NFS_VALUE(index) (nfsnames[index].n_value)
# define NUMBER_NFS_NLISTS 4
# define NLIST_NFS_RCSTAT  0
# define NLIST_NFS_RSSTAT  1
# define NLIST_NFS_CLSTAT  2
# define NLIST_NFS_SVSTAT  3

struct nlist nfsnames[] = {
    {"rcstat", 0, 0, 0, 0, 0},
    {"rsstat", 0, 0, 0, 0, 0},
    {"clstat", 0, 0, 0, 0, 0},
    {"svstat", 0, 0, 0, 0, 0},
    {NULL,     0, 0, 0, 0, 0},
};

#endif

extern char *progname;



void
init_nfs()
{
#if USE_NFS_CNTL == 1

    void * handle;
# ifdef DO_SYSCONFIG
    struct cfg_load cload;

    cload.path = NFS_EXT_PATH;
    cload.kmid = 0;
# endif

    if (show_nfs || logmode) {
# ifdef DO_SYSCONFIG
        /* 
         * Check if NFS kernel extension is present
         *
         * Now dlopen etc works, do we still need the sysconfig check?
         */
        if (sysconfig (SYS_QUERYLOAD, &cload, sizeof (cload)) != 0) {
            perror("init_nfs: sysconfig NFS query");
            show_nfs = 0;
            have_nfs = 0;
            return;
        }

        if (!cload.kmid) {
            fprintf(stderr, "init_nfs: NFS Kernel module not loaded\n");
            show_nfs = 0;
            have_nfs = 0;
            return;
        }
# endif

        /* 
         * Get the call address for the nfs_cntl system call
         */
        if ((handle = dlopen("/unix", RTLD_NOW | RTLD_GLOBAL)) == NULL) {
            fprintf(stderr, "%s: init_nfs: dlopen: can't load %s (%s)\n", 
                            progname, "/unix", dlerror());
            show_nfs = 0;
            have_nfs = 0;
            return;
        }
        my_nfs_cntl = dlsym( handle, "nfs_cntl");
        if (my_nfs_cntl == NULL) {
            fprintf(stderr, "%s: init_nfs: dlsym: can't find %s (err %s)\n",
                            progname, "nfs_cntl", dlerror());
            show_nfs = 0;
            have_nfs = 0;
        }
        dlclose(handle);
    }

#else

    if (show_nfs) {
        if (knlist(nfsnames, NUMBER_NFS_NLISTS, sizeof(struct nlist)) == -1) {
            perror("get_nfsstat: nfs not configured");
            show_nfs = 0;
            have_nfs = 0;
            /* bail out before we try getkmemdata      */
            return;
        }
    }
#endif

    return;

} /* init_nfs */


void
get_nfsstat(t_nfs_stats *nfsstat)
{

#if USE_NFS_CNTL == 1
    my_nfs_cntl(NFS_CNTL_GET_ALL_STAT, nfsstat, NFS_CNTL_ALL_SIZE);

#else

    getkmemdata((char *) &nfsstat->rcstat, sizeof(t_rcstat),
                (caddr_t) NFS_VALUE(NLIST_NFS_RCSTAT));
    getkmemdata((char *) &nfsstat->rsstat, sizeof(t_rsstat),
                (caddr_t) NFS_VALUE(NLIST_NFS_RSSTAT));
    getkmemdata((char *) &nfsstat->clstat, sizeof(t_clstat),
                (caddr_t) NFS_VALUE(NLIST_NFS_CLSTAT));
    getkmemdata((char *) &nfsstat->svstat, sizeof(t_svstat),
                (caddr_t) NFS_VALUE(NLIST_NFS_SVSTAT));

#endif

    return;

} /* get_nfsstat */
