/* -- AIX/6000 System monitor 
**
**     fps.c
**
** Copyright (c) 1991-1995 Jussi Maki, All Rights Reserved.
** Copyright (c) 1993-2001 Marcel Mol, All Rights Reserved.
** NON-COMMERCIAL USE ALLOWED. YOU ARE FREE TO DISTRIBUTE
** THIS PROGRAM AND MODIFY IT AS LONG AS YOU KEEP ORIGINAL
** COPYRIGHTS.
*/

/* Fast ps like utility "fps" -- Jussi Maki, Center for Scientific Computing,
 * Finland, jmaki@csc.fi,  February 21, 1992 - July 15, 1996
 * compile this program in AIX with eg. command:
 *   cc -o fps fps.c
 */

#include <stdio.h>
#include <ctype.h>
#include <procinfo.h>
#include <locale.h>
#include <sys/time.h>
#include "config.h"

#define NPROCS 10000

/* system call to get process table */
extern getproc(struct procinfo *procinfo, int nproc, int sizproc);
    /* procinfo:   pointer to array of procinfo struct */
    /* nproc:      number of user procinfo struct */
    /* sizproc:    size of expected procinfo structure */

/* system call to get program arguments */
extern getargs(struct procinfo *procinfo, int plen, char *args, int alen);
    /* procinfo:   pointer to array of procinfo struct */
    /* plen:       size of expected procinfo struct */
    /* args:       pointer to user array for arguments */
    /* alen:       size of expected argument array */

/* system call to get user-area variables according to process */
extern getuser(struct procinfo *procinfo, int plen, void *user, int ulen);
    /* procinfo:   ptr to array of procinfo struct */
    /* plen:       size of expected procinfo struct */
    /* user:       ptr to array of userinfo struct, OR user */
    /* ulen:       size of expected userinfo struct */

struct procsortinfo {
     int    index;            /* index to previous procinfo-array */
     double deltacputime;
     double cputime;
};

/* memory allocation doesn't really happen until it is touched
 * so these large arrays don't waste memory in AIX 3
 */
struct procinfo proc1[NPROCS];
struct userinfo user1[NPROCS];

/* chars which contain process status symbols */
/* char statch[] = {'N', 'S', '?', 'R', 'T', 'Z', 'S'}; */
char statch[] = {'0', 'S', '2', 'R', 'I', 'Z', 's', 'r', 'T'};
/*
 * 0 = N Available slot
 * 1 = ? Sleep?
 * 2 = ?
 * 3 = ? Running?
 * 4 = I Process is created
 * 5 = Z Process is Zombie/Dying
 * 6 = S Process is Stopped
 * 7 = R Process is Running/Active
 * 8 = T Process is Swapped
 */



#define SORTBYCPUTIME 1
#define SORTBYCPUPERCENT 2

FILE *logfile;

int show_allprocs = 0, show_mem;

int proc_to_watch = -1;
int sleep_time = 1;
int count = 1;
int numlines;
char * progname;


/*
 * Function Delcarations
 */
void   usage();
double walltime();
void   print_one_proc(pid_t pid, struct procinfo *proc,
                                 struct userinfo *user, int nproc);
void   printprocs(struct procinfo *proc, struct userinfo *user, int nproc);
int    getprocessinfo(struct procinfo *procinfo, struct userinfo *userinfo);



main(int argc, char **argv)
{
    int result;
    int i;
    int nproc1;
    double cpusum;

    progname = *argv;
    while (*++argv) {
        if (**argv != '-')
            usage();
        switch(argv[0][1]) {
            case 'a': show_allprocs = 1;
                      break;
            case 'm': show_mem = 1;
                      break;
            case 'p': proc_to_watch = atoi(*++argv);
                      numlines = atoi(termdef(fileno(stdout), 'l'));
                      if (numlines == 0) numlines = 20;
                      if (count == 1)
                          count = 2;
                      break;
            case 'c': if (isalnum(argv[1][0])) {
		          count = atoi(argv[1]);
		          ++argv;
	              }
                      else
	                 count = -1;
                      break;
            case 's': sleep_time = atoi(*++argv);
                      break;
            default:  usage();
        }
    }

    do {
        nproc1 = getprocessinfo(proc1, user1);
        if (proc_to_watch >= 0) 
            print_one_proc(proc_to_watch, proc1, user1, nproc1);
        else
            printprocs(proc1, user1, nproc1);
        if (--count)
            sleep(sleep_time);
    } while (count);

    exit(0);

} /* main */



void
usage()
{

    fprintf(stderr, "Usage: %s [-m] [-a] [-p pid] [-s sleep] [-count [num]]\n"
                    "\n\tfps is a fast ps which shows processes without "
                    "interpreting uids\n", progname);

    exit(1);

} /* usage */



double
walltime()
{
    struct timeval tv;
    struct timezone tz;

    gettimeofday(&tv, &tz);

    return(tv.tv_sec + tv.tv_usec * 1.0e-6);

} /* walltime */



void
print_one_proc(pid_t pid, struct procinfo *proc, struct userinfo *user,
               int nproc)
{
    static int runs = 0;
    static double        old_walltime,  new_walltime;
    static double        old_cputime,   new_cputime;
    static unsigned long old_pagefault, new_pagefault;
    int i;

    for (i = 0; i < nproc; i++)
        if (proc[i].pi_pid == pid)
            break;
    if (i >= nproc) {
        fprintf(stderr, "No such process.\n"
                        "Maybe NPROCS is to low, then recompile me...\n");
        exit(0);
    }
    old_walltime  = new_walltime;
    old_cputime   = new_cputime;
    old_pagefault = new_pagefault;
    new_walltime  = walltime();
    new_cputime   = user[i].ui_ru.ru_utime.tv_sec +
                    user[i].ui_ru.ru_stime.tv_sec +
                    (user[i].ui_ru.ru_utime.tv_usec +
                     user[i].ui_ru.ru_stime.tv_usec) * 1.0e-6;
    new_pagefault = proc[i].pi_majflt;
    if (numlines > 2 && runs % (numlines - 2) == 0 ) {
        printf("#WATCH PID %d\n", pid);
        printf("TOTCPU     CPU%%  RESMEM  VIRMEM PGFLTS WAITCHAN WAITTYPE\n");
    }
    /*
     * The first time we are called, printed will become 2 from above if.
     * But the first time there is not enough date to print results yet.
     */
    if (runs > 0) {
        printf("%-8.1f %5.2f%% %6.2fM %6.2fM %6d %8x %8x\n",
	       new_cputime,
	       (new_cputime - old_cputime)/(new_walltime - old_walltime)*100.0,
	       (user[i].ui_trss + user[i].ui_drss) * 4 / 1024.0,
	       (user[i].ui_tsize / 1024.0 + user[i].ui_dvm * 4) / 1024.0,
	       new_pagefault - old_pagefault,
	       proc[i].pi_wchan, 
	       proc[i].pi_wtype);
    }
    runs++;

    return;

} /* print_one_proc */



/*
 * CPU -field shows the latest CPU-time usage accumalated in AIX scheduler
 * PRI -field is the current priority used in AIX scheduler
 * USER- and SYSTIME show process cputime by usertime and systemtime
 */

void
printprocs(struct procinfo *proc, struct userinfo *user, int nproc)
{
    int i;

    if (show_mem) 
        printf("   PID   PPID PGRPID   UID  SIZE  RES STAT  USER-  SYSTIME "
               "CHILDS COMMAND\n");
    else
        printf("   PID   PPID PGRPID   UID PRI NICE CPU STAT USER-  SYSTIME "
               "CHILDS COMMAND\n");
    for (i = 0; i < nproc; i++) {
        if (proc[i].pi_cpu != 0 || show_allprocs) {
	    printf("%6d %6d %6d %5d ", proc[i].pi_pid,  proc[i].pi_ppid,
                                       proc[i].pi_pgrp, proc[i].pi_uid);
	    if (show_mem) 
	        printf("%4.1fM %4.1fM ",
		      (user[i].ui_tsize / 1024.0 + user[i].ui_dvm * 4) / 1024.0,
		      (user[i].ui_trss + user[i].ui_drss) * 4 / 1024.0);
	    else 
	        printf("%3d %4d %3d ", proc[i].pi_pri, proc[i].pi_nice - 20,
	                               proc[i].pi_cpu);
	    if (proc[i].pi_stat != SZOMB) {
	        printf("%c%2s %8.2f %8.2f %6.2f %s",
	               statch[proc[i].pi_stat],
                       user[i].ui_ttyp ? "fg" : "bg",
	               user[i].ui_ru.ru_utime.tv_sec +
                           user[i].ui_ru.ru_utime.tv_usec * 1e-6,
	               user[i].ui_ru.ru_stime.tv_sec +
                           user[i].ui_ru.ru_stime.tv_usec * 1e-6,
	               user[i].ui_cru.ru_utime.tv_sec +
   	                 user[i].ui_cru.ru_stime.tv_sec +
                           (user[i].ui_cru.ru_utime.tv_usec +
                            user[i].ui_cru.ru_stime.tv_usec) * 1e-6,
	               user[i].ui_comm);
	        if (proc[i].pi_flag & SKPROC)
                    printf(" (kproc)");
	    }
            else
	        printf("%6d %6d (ZOMBIE)", proc[i].pi_utime, proc[i].pi_stime);
	    putchar('\n');
        }
    }

    return;

} /* printprocs */



int
getprocessinfo(struct procinfo *procinfo, struct userinfo *userinfo)
{
    int nproc;
    char *swappername = "swapper";
    int i;

    /* get the whole process table */
    nproc = getproc(procinfo, NPROCS, sizeof(struct procinfo));
    for (i = 0; i < nproc; i++) /* get each user-area entrys by process */
        getuser(&procinfo[i], sizeof(struct procinfo),
	        &userinfo[i], sizeof(struct userinfo));
    strcpy(userinfo[0].ui_comm, swappername); /* 1st process is always pid 0 */

    return nproc;

} /* getprocessinfo */
